import { Room, RoomEvent, Track } from 'livekit-client';

window.addEventListener('DOMContentLoaded', async () => {
    const videoContainer = document.getElementById('video');
    if (!videoContainer) return;

    const data = document.getElementById('livekit-data').dataset;
    const serverUrl = data.server;
    const token = data.token;
    const roomName = data.room;

    const room = new Room({
        adaptiveStream: true,
        dynacast: true,
    });

    let localVideoEnabled = true;
    let localAudioEnabled = true;
    let isScreenSharing = false;

    console.log('🚀 Connexion à LiveKit...');

    // ==================== FONCTIONS UTILITAIRES ====================

    function addMessage(sender, text, isSystem = false) {
        const container = document.getElementById('messages');
        if (!container) return;

        const div = document.createElement('div');
        if (isSystem) {
            div.className = 'text-center text-xs text-gray-400 py-2';
            div.textContent = text;
        } else {
            div.className = 'bg-gray-700 p-3 rounded-lg mb-2';
            div.innerHTML = `
                <div class="font-bold text-sm text-blue-400">${sender}</div>
                <div class="text-sm mt-1">${text}</div>
            `;
        }
        container.appendChild(div);
        container.scrollTop = container.scrollHeight;
    }

    function updateParticipantCount() {
        const count = room.participants.size + 1;
        const el = document.getElementById('participant-count');
        if (el) el.textContent = `${count} participant${count > 1 ? 's' : ''}`;
    }

    function createVideoElement(track, participantName, isLocal = false) {
        const container = document.createElement('div');
        container.className = 'relative bg-gray-800 rounded-lg overflow-hidden';
        container.style.minHeight = '250px';

        const video = track.attach();
        video.className = 'w-full h-full object-cover';
        video.style.transform = isLocal ? 'scaleX(-1)' : 'none';

        const label = document.createElement('div');
        label.className = 'absolute bottom-2 left-2 bg-black bg-opacity-70 px-3 py-1 rounded text-sm';
        label.textContent = participantName + (isLocal ? ' (Vous)' : '');

        container.appendChild(video);
        container.appendChild(label);
        container.id = 'video-' + participantName.replace(/\s/g, '-');

        return container;
    }

    // ==================== ÉVÉNEMENTS ROOM ====================

    room.on(RoomEvent.Connected, async () => {
        console.log('✅ Connecté !');
        addMessage('Système', 'Vous êtes connecté à la salle', true);

        // Activer caméra et micro
        try {
            await room.localParticipant.setCameraEnabled(true);
            await room.localParticipant.setMicrophoneEnabled(true);

            // Afficher ma vidéo
            const videoTrack = room.localParticipant.videoTrackPublications.values().next().value;
            if (videoTrack && videoTrack.track) {
                const el = createVideoElement(videoTrack.track, 'Vous', true);
                videoContainer.appendChild(el);
            }

            updateParticipantCount();
        } catch (err) {
            console.error('Erreur média:', err);
            addMessage('Système', '⚠️ Impossible d\'activer la caméra/micro', true);
        }
    });

    room.on(RoomEvent.TrackSubscribed, (track, publication, participant) => {
        console.log('📹 Nouvelle piste:', track.kind);

        if (track.kind === Track.Kind.Video) {
            const el = createVideoElement(track, participant.identity);
            videoContainer.appendChild(el);
        }
    });

    room.on(RoomEvent.TrackUnsubscribed, (track, publication, participant) => {
        if (track.kind === Track.Kind.Video) {
            const el = document.getElementById('video-' + participant.identity.replace(/\s/g, '-'));
            if (el) el.remove();
        }
    });

    room.on(RoomEvent.ParticipantConnected, (participant) => {
        console.log('👤 Participant:', participant.identity);
        addMessage('Système', `${participant.identity} a rejoint`, true);
        updateParticipantCount();
    });

    room.on(RoomEvent.ParticipantDisconnected, (participant) => {
        console.log('👋 Départ:', participant.identity);
        addMessage('Système', `${participant.identity} est parti`, true);
        updateParticipantCount();

        const el = document.getElementById('video-' + participant.identity.replace(/\s/g, '-'));
        if (el) el.remove();
    });

    room.on(RoomEvent.DataReceived, (payload, participant) => {
        const text = new TextDecoder().decode(payload);
        addMessage(participant?.identity || 'Anonyme', text);
    });

    // ==================== CONTRÔLES ====================

    // Toggle Micro
    document.getElementById('toggle-audio')?.addEventListener('click', async () => {
        localAudioEnabled = !localAudioEnabled;
        await room.localParticipant.setMicrophoneEnabled(localAudioEnabled);

        const btn = document.getElementById('toggle-audio');
        btn.classList.toggle('bg-red-600', !localAudioEnabled);
        btn.classList.toggle('bg-gray-700', localAudioEnabled);

        console.log('🎤', localAudioEnabled ? 'ON' : 'OFF');
    });

    // Toggle Vidéo
    document.getElementById('toggle-video')?.addEventListener('click', async () => {
        localVideoEnabled = !localVideoEnabled;
        await room.localParticipant.setCameraEnabled(localVideoEnabled);

        const btn = document.getElementById('toggle-video');
        btn.classList.toggle('bg-red-600', !localVideoEnabled);
        btn.classList.toggle('bg-gray-700', localVideoEnabled);

        console.log('📹', localVideoEnabled ? 'ON' : 'OFF');
    });

    // Partage d'écran
    document.getElementById('share-screen')?.addEventListener('click', async () => {
        try {
            isScreenSharing = !isScreenSharing;
            await room.localParticipant.setScreenShareEnabled(isScreenSharing);

            const btn = document.getElementById('share-screen');
            btn.classList.toggle('bg-green-600', isScreenSharing);
            btn.classList.toggle('bg-gray-700', !isScreenSharing);

            addMessage('Système', isScreenSharing ? '🖥️ Partage écran activé' : '🖥️ Partage écran désactivé', true);
            console.log('🖥️', isScreenSharing ? 'ON' : 'OFF');
        } catch (err) {
            console.error('Erreur partage écran:', err);
            addMessage('Système', '❌ Erreur partage écran', true);
        }
    });

    // Toggle Chat
    let chatOpen = false;
    document.getElementById('toggle-chat')?.addEventListener('click', () => {
        chatOpen = !chatOpen;
        const sidebar = document.getElementById('chat-sidebar');
        sidebar.classList.toggle('hidden', !chatOpen);
        console.log('💬 Chat', chatOpen ? 'ouvert' : 'fermé');
    });

    // Envoyer message
    document.getElementById('message-form')?.addEventListener('submit', (e) => {
        e.preventDefault();
        const input = document.getElementById('message-input');
        const text = input.value.trim();

        if (text) {
            const data = new TextEncoder().encode(text);
            room.localParticipant.publishData(data, { reliable: true });
            addMessage('Vous', text);
            input.value = '';
            console.log('📤 Message envoyé');
        }
    });

    // Quitter
    document.getElementById('leave-room')?.addEventListener('click', () => {
        if (confirm('Quitter la salle ?')) {
            room.disconnect();
            window.location.href = '/student/livekit';
        }
    });

    // ==================== CONNEXION ====================

    try {
        await room.connect(serverUrl, token);
        console.log('🔗 Connecté au serveur');
    } catch (err) {
        console.error('❌ Erreur connexion:', err);
        alert('Impossible de se connecter: ' + err.message);
    }
});
