<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;

class User extends Authenticatable
{
    use HasFactory, Notifiable;

    /**
     * Les attributs qui peuvent être assignés en masse.
     */
    protected $fillable = [
        'first_name',
        'last_name',
        'username',
        'email',
        'password',
        'verification_code',
        'phone',
        'second_phone',
        'date_of_birth',
        'gender',
        'governorate',
        'city',
        'postal_code',
        'street_address',
        'role',
    ];

    /**
     * Les attributs à cacher dans les tableaux ou les réponses JSON.
     */
    protected $hidden = [
        'password',
        'remember_token',
        'verification_code',
    ];

    /**
     * Les attributs avec des conversions de type.
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        'date_of_birth' => 'date',
        'password' => 'hashed',
    ];

    // ==========================================
    // 🔹 RELATIONS PERSONNALISÉES
    // ==========================================

    /**
     * Conversations avec Nova (IA intégrée)
     */
    public function novaConversations()
    {
        return $this->hasMany(NovaConversation::class);
    }

    /**
     * Dernière conversation Nova
     */
    public function latestNovaConversation()
    {
        return $this->hasOne(NovaConversation::class)->latestOfMany('last_message_at');
    }

    /**
     * Groupes de l’utilisateur (relation pivot)
     */
    public function groups()
    {
        return $this->belongsToMany(Group::class)
            ->withPivot('role')
            ->withTimestamps();
    }

    /**
     * Sessions live où l’utilisateur est professeur
     */
    public function teachingSessions()
    {
        return $this->hasMany(LiveSession::class, 'teacher_id');
    }

    /**
     * Tâches associées à l’utilisateur (To-do)
     */
    public function tasks()
    {
        return $this->hasMany(StudentTask::class);
    }

    /**
     * Feedbacks laissés par l’utilisateur
     */
    public function feedbacks()
    {
        return $this->hasMany(Feedback::class);
    }

    /**
     * Présences aux sessions
     */
    public function attendances()
    {
        return $this->hasMany(SessionAttendance::class);
    }

    public function wallet()
{
    return $this->hasOne(Wallet::class);
}

public function walletTransactions()
{
    return $this->hasMany(WalletTransaction::class);
}

public function walletNotifications()
{
    return $this->hasMany(WalletNotification::class);
}
}
