<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;

class NovaConversation extends Model
{
    protected $fillable = [
        'user_id',
        'title',
        'last_message_at',
    ];

    protected $casts = [
        'last_message_at' => 'datetime',
    ];

    /**
     * Relation avec l'utilisateur
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Relation avec les messages
     */
    public function messages(): HasMany
    {
        return $this->hasMany(NovaMessage::class, 'conversation_id');
    }

    /**
     * Obtenir les messages ordonnés
     */
    public function orderedMessages()
    {
        return $this->messages()->orderBy('created_at', 'asc');
    }

    /**
     * Obtenir le dernier message de l'utilisateur
     */
    public function lastUserMessage(): HasOne
    {
        return $this->hasOne(NovaMessage::class, 'conversation_id')
                    ->where('role', 'user')
                    ->latest('created_at');
    }

    /**
     * Générer un titre automatique basé sur le premier message
     */
    public function generateTitle(): void
    {
        if (!$this->title) {
            $firstMessage = $this->messages()->where('role', 'user')->first();
            if ($firstMessage) {
                $this->title = \Illuminate\Support\Str::limit($firstMessage->content, 50);
                $this->save();
            }
        }
    }

    /**
     * Mettre à jour le timestamp du dernier message
     */
    public function updateLastMessageTime(): void
    {
        $this->last_message_at = now();
        $this->save();
    }
}
