<?php

namespace App\Http\Controllers\Teacher;

use App\Http\Controllers\Controller;
use App\Models\Group;
use App\Models\Meeting;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Schema;

class LiveSessionteacherController extends Controller
{


    /**
     * 📅 Affiche la page principale (vue enseignant)
     */
public function index()
{
    $user = Auth::user();

    // Récupérer les groupes du teacher
    $groups = Group::whereHas('users', function ($q) {
        $q->where('users.id', Auth::id())
          ->where('users.role', 'teacher');
    })->get();

    // Meetings Zoom du teacher
    $meetings = Meeting::where('user_id', $user->id)
        ->with('group')
        ->orderBy('start_time', 'asc')
        ->get();

        $calendarSessions = \App\Models\LiveSession::where('user_id', $user->id)
    ->orderBy('start_time')
    ->get();

    // 🔥 Ajouter les tasks (même table que student)
    $tasks = \App\Models\StudentTask::where('user_id', $user->id)
        ->orderBy('completed')
        ->orderByDesc('created_at')
        ->get();

    return view('teacher.live-sessions.index', compact('user', 'groups', 'meetings', 'tasks', 'calendarSessions'));
}


    /**
     * ➕ Créer un meeting Zoom pour un groupe spécifique
     */
    public function createZoomMeeting($groupId)
    {
        try {
            $token = $this->getZoomAccessToken();

            $group = Group::findOrFail($groupId);
            $teacher = Auth::user();

            $response = Http::withToken($token)->post('https://api.zoom.us/v2/users/me/meetings', [
                'topic' => 'Live Class - ' . $group->name,
                'type' => 2,
                'start_time' => now()->addMinutes(5)->toIso8601String(),
                'duration' => 60,
                'timezone' => 'Africa/Tunis',
                'settings' => [
                    'host_video' => true,
                    'participant_video' => true,
                    'waiting_room' => false,
                    'join_before_host' => true,
                ],
            ]);

            if ($response->failed()) {
                return back()->with('error', 'Zoom API Error: ' . $response->body());
            }

            $data = $response->json();

            Meeting::create([
                'user_id' => $teacher->id,
                'group_id' => $groupId,
                'topic' => $data['topic'],
                'zoom_meeting_id' => $data['id'],
                'start_url' => $data['start_url'],
                'join_url' => $data['join_url'],
                'password' => $data['password'] ?? null,
                'start_time' => date('Y-m-d H:i:s', strtotime($data['start_time'])),
                'end_time' => now()->addHour(),
            ]);

            return back()->with('success', "Zoom meeting created for {$group->name}");

        } catch (\Exception $e) {
            return back()->with('error', $e->getMessage());
        }
    }

    /**
     * 🔑 Récupère le token OAuth Zoom
     */
    private function getZoomAccessToken()
    {
        $clientId = env('ZOOM_API_CLIENT_ID');
        $clientSecret = env('ZOOM_API_CLIENT_SECRET');
        $accountId = env('ZOOM_ACCOUNT_ID');

        $response = Http::asForm()
            ->withBasicAuth($clientId, $clientSecret)
            ->post('https://zoom.us/oauth/token', [
                'grant_type' => 'account_credentials',
                'account_id' => $accountId,
            ]);

        if ($response->failed()) {
            throw new \Exception('Unable to get Zoom access token');
        }

        return $response->json()['access_token'];
    }
}
