<?php

namespace App\Http\Controllers\Student;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Firebase\JWT\JWT;
use Illuminate\Support\Facades\Schema;


class ZoomController extends Controller
{
    /**
     * 📋 Afficher la liste des meetings
     */
    public function index()
    {
        $meetings = DB::table('meetings')
            ->where('user_id', Auth::id())
            ->orderBy('start_time', 'desc')
            ->get();

        return view('student.zoom.index', compact('meetings'));
    }

    /**
     * ➕ Créer un meeting via l'API Zoom (utilise OAuth credentials)
     */
    public function create(Request $request)
    {
        try {
            \Log::info('=== CREATING ZOOM MEETING (teacher UI) ===');

            $user = Auth::user();
            $token = $this->getZoomAccessToken();

            $request->validate([
                'day'        => 'required|string',
                'start_hour' => 'required|string',
                'duration'   => 'required|integer',
                'group_id'   => 'nullable|integer',
                'topic'      => 'required|string|max:255',
            ]);

            $weekStart = \Carbon\Carbon::now()->startOfWeek();

            $dayIndex = [
                'monday'    => 0,
                'tuesday'   => 1,
                'wednesday' => 2,
                'thursday'  => 3,
                'friday'    => 4,
                'saturday'  => 5,
                'sunday'    => 6,
            ];

            $dayDate = $weekStart->copy()->addDays($dayIndex[$request->day]);

            $start_time = \Carbon\Carbon::parse($dayDate->format('Y-m-d') . ' ' . $request->start_hour);
            $duration = intval($request->duration);
            $end_time = $start_time->copy()->addMinutes($duration);

            \Log::info('Computed Start: ' . $start_time);
            \Log::info('Computed End: ' . $end_time);

            $response = Http::withToken($token)->post('https://api.zoom.us/v2/users/me/meetings', [
                'topic' => $request->topic,
                'type' => 2,
                'start_time' => $start_time->toIso8601String(),
                'duration' => $request->duration,
                'timezone' => 'Africa/Tunis',
                'settings' => [
                    'host_video' => true,
                    'participant_video' => true,
                    'waiting_room' => false,
                    'join_before_host' => true,
                ]
            ]);

            if ($response->failed()) {
                \Log::error('Zoom API Error', ['response' => $response->body()]);
                return back()->with('error', 'Erreur API Zoom: ' . $response->body());
            }

            $data = $response->json();

            $meeting = [
                'user_id'          => $user->id,
                'group_id'         => $request->group_id,
                'topic'            => $data['topic'],
                'zoom_meeting_id'  => $data['id'],
                'start_url'        => $data['start_url'],
                'join_url'         => $data['join_url'],
                'start_time'       => $start_time->format('Y-m-d H:i:s'),
                'end_time'         => $end_time->format('Y-m-d H:i:s'),
                'created_at'       => now(),
                'updated_at'       => now(),
            ];

            if (Schema::hasColumn('meetings', 'password')) {
                $meeting['password'] = $data['password'] ?? null;
            }

            DB::table('meetings')->insert($meeting);

            return redirect()->back()->with('success', 'Zoom meeting created successfully!');

        } catch (\Exception $e) {
            \Log::error('Zoom Creation Error: ' . $e->getMessage());
            return back()->with('error', $e->getMessage());
        }
    }

    /**
     * ✏️ Mettre à jour un meeting
     */
    public function update(Request $request, $id)
    {
        try {
            \Log::info('=== UPDATING MEETING ===', ['id' => $id]);

            $meeting = DB::table('meetings')
                ->where('id', $id)
                ->where('user_id', Auth::id())
                ->first();

            if (!$meeting) {
                return response()->json(['error' => 'Meeting not found'], 404);
            }

            $updateData = [
                'topic'      => $request->input('topic', $meeting->topic),
                'updated_at' => now(),
            ];

            // Description (si la colonne existe)
            if (Schema::hasColumn('meetings', 'description')) {
                $updateData['description'] = $request->input('description');
            }

            // Subtasks (si la colonne existe)
            if (Schema::hasColumn('meetings', 'subtasks')) {
                $subtasks = $request->input('subtasks', []);
                $updateData['subtasks'] = json_encode($subtasks);
            }

            // Mise à jour des heures si fournies
            if ($request->has('start_time') && $request->has('end_time')) {
                $currentDate = \Carbon\Carbon::parse($meeting->start_time)->format('Y-m-d');

                $newStartTime = \Carbon\Carbon::parse($currentDate . ' ' . $request->input('start_time'));
                $newEndTime = \Carbon\Carbon::parse($currentDate . ' ' . $request->input('end_time'));

                $updateData['start_time'] = $newStartTime->format('Y-m-d H:i:s');
                $updateData['end_time'] = $newEndTime->format('Y-m-d H:i:s');

                // Optionnel : mettre à jour aussi sur Zoom API
                try {
                    $token = $this->getZoomAccessToken();
                    Http::withToken($token)->patch("https://api.zoom.us/v2/meetings/{$meeting->zoom_meeting_id}", [
                        'topic' => $updateData['topic'],
                        'start_time' => $newStartTime->toIso8601String(),
                        'duration' => $newStartTime->diffInMinutes($newEndTime),
                    ]);
                } catch (\Exception $e) {
                    \Log::warning('Could not update Zoom meeting: ' . $e->getMessage());
                }
            }

            DB::table('meetings')
                ->where('id', $id)
                ->update($updateData);

            \Log::info('✅ Meeting updated successfully');

            return response()->json(['success' => true]);

        } catch (\Exception $e) {
            \Log::error('Meeting Update Error: ' . $e->getMessage());
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * 🗑️ Supprimer un meeting
     */
    public function destroy($id)
    {
        try {
            \Log::info('=== DELETING MEETING ===', ['id' => $id]);

            $meeting = DB::table('meetings')
                ->where('id', $id)
                ->where('user_id', Auth::id())
                ->first();

            if (!$meeting) {
                return response()->json(['error' => 'Meeting not found'], 404);
            }

            // Supprimer aussi sur Zoom API
            try {
                $token = $this->getZoomAccessToken();
                Http::withToken($token)->delete("https://api.zoom.us/v2/meetings/{$meeting->zoom_meeting_id}");
                \Log::info('✅ Zoom meeting deleted from Zoom API');
            } catch (\Exception $e) {
                \Log::warning('Could not delete from Zoom API: ' . $e->getMessage());
            }

            // Supprimer de la base de données
            DB::table('meetings')->where('id', $id)->delete();

            \Log::info('✅ Meeting deleted from database');

            return response()->json(['success' => true]);

        } catch (\Exception $e) {
            \Log::error('Meeting Delete Error: ' . $e->getMessage());
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * 🔑 Obtenir un access token OAuth pour l'API REST
     */
    private function getZoomAccessToken()
    {
        $clientId = env('ZOOM_API_CLIENT_ID');
        $clientSecret = env('ZOOM_API_CLIENT_SECRET');
        $accountId = env('ZOOM_ACCOUNT_ID');

        \Log::info('=== OBTAINING OAUTH TOKEN ===');

        if (empty($clientId) || empty($clientSecret) || empty($accountId)) {
            \Log::error('❌ OAuth credentials missing in .env');
            throw new \Exception('Configuration OAuth Zoom incomplète.');
        }

        try {
            $response = Http::asForm()
                ->withBasicAuth($clientId, $clientSecret)
                ->post('https://zoom.us/oauth/token', [
                    'grant_type' => 'account_credentials',
                    'account_id' => $accountId,
                ]);

            if ($response->failed()) {
                throw new \Exception('Impossible d\'obtenir le token Zoom.');
            }

            $data = $response->json();

            if (!isset($data['access_token'])) {
                throw new \Exception('Token absent dans la réponse Zoom');
            }

            return $data['access_token'];

        } catch (\Exception $e) {
            \Log::error('❌ Exception in getZoomAccessToken', ['message' => $e->getMessage()]);
            throw $e;
        }
    }

    /**
     * 🎨 Générer une signature JWT pour le Meeting SDK
     */
    public function generateSignature(Request $request)
    {
        $meetingNumber = $request->input('meetingNumber');
        $role = $request->input('role', 0);

        $sdkKey = env('ZOOM_SDK_KEY');
        $sdkSecret = env('ZOOM_SDK_SECRET');

        if (empty($sdkKey) || empty($sdkSecret)) {
            return response()->json(['error' => 'Configuration SDK Zoom manquante.'], 500);
        }

        $iat = time();
        $exp = $iat + 60 * 60 * 2;

        $payload = [
            'appKey' => $sdkKey,
            'sdkKey' => $sdkKey,
            'mn' => $meetingNumber,
            'role' => intval($role),
            'iat' => $iat,
            'exp' => $exp,
            'tokenExp' => $exp
        ];

        try {
            $signature = JWT::encode($payload, $sdkSecret, 'HS256');
            return response()->json(['signature' => $signature]);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Erreur génération signature'], 500);
        }
    }

    /**
     * 🚪 Rejoindre un meeting
     */
    public function join($meetingId)
    {
        $meeting = DB::table('meetings')
            ->where('zoom_meeting_id', $meetingId)
            ->where('user_id', Auth::id())
            ->first();

        if (!$meeting) {
            return redirect()->route('student.zoom.index')
                ->with('error', 'Réunion introuvable ou accès refusé');
        }

        if (now()->greaterThan($meeting->end_time)) {
            return redirect()->route('student.zoom.index')
                ->with('error', 'Cette réunion est terminée');
        }

        $sdkKey = config('services.zoom.sdk_key', env('ZOOM_SDK_KEY'));

        return view('student.zoom.join', compact('meeting', 'sdkKey'));
    }

    /**
     * 🎬 Démarrer un meeting (Teacher)
     */
    public function start($meetingId)
    {
        $meeting = DB::table('meetings')
            ->where('zoom_meeting_id', $meetingId)
            ->where('user_id', Auth::id())
            ->first();

        if (!$meeting) {
            return redirect()->route('teacher.live-sessions.index')
                ->with('error', 'Réunion introuvable ou accès refusé.');
        }

        $sdkKey = config('services.zoom.sdk_key', env('ZOOM_SDK_KEY'));

        return view('student.zoom.join', [
            'meeting' => $meeting,
            'sdkKey' => $sdkKey,
            'role' => 1
        ]);
    }

 public function uploadAttachment(Request $request, $id)
{
    try {
        \Log::info('=== UPLOAD ATTACHMENT ===');
        \Log::info('Meeting ID: ' . $id);
        \Log::info('Content-Type: ' . $request->header('Content-Type'));
        \Log::info('Has file (file): ' . ($request->hasFile('file') ? 'yes' : 'no'));
        \Log::info('Files: ' . json_encode(array_keys($_FILES)));
        \Log::info('$_FILES: ' . json_encode($_FILES));

        // Essayer de récupérer le fichier de différentes manières
        $file = $request->file('file') ?? $request->files->get('file') ?? null;

        if (!$file && isset($_FILES['file'])) {
            \Log::info('Using $_FILES fallback');
            $file = $request->file('file');
        }

        if (!$file) {
            \Log::error('No file found in request');
            return response()->json(['error' => 'No file uploaded', 'debug' => $_FILES], 400);
        }

        // Vérifier le type de fichier
        $allowedMimes = [
            'image/jpeg', 'image/png', 'image/gif', 'image/webp',
            'application/pdf',
            'application/msword',
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'application/vnd.ms-excel',
            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'application/vnd.ms-powerpoint',
            'application/vnd.openxmlformats-officedocument.presentationml.presentation'
        ];

        if (!in_array($file->getMimeType(), $allowedMimes)) {
            return response()->json(['error' => 'File type not allowed: ' . $file->getMimeType()], 400);
        }

        if ($file->getSize() > 10 * 1024 * 1024) {
            return response()->json(['error' => 'File too large (max 10MB)'], 400);
        }

        $meeting = DB::table('meetings')
            ->where('id', $id)
            ->first();

        if (!$meeting) {
            return response()->json(['error' => 'Meeting not found'], 404);
        }

        // Upload
        $filename = time() . '_' . preg_replace('/[^a-zA-Z0-9._-]/', '', $file->getClientOriginalName());
        $path = $file->storeAs('meetings/' . $id, $filename, 'public');

        $attachments = json_decode($meeting->attachments, true) ?? [];

        $attachments[] = [
            'name' => $file->getClientOriginalName(),
            'path' => $path,
            'url' => asset('storage/' . $path),
            'type' => $this->getFileType($file->getMimeType()),
            'mime' => $file->getMimeType(),
            'size' => $file->getSize(),
        ];

        DB::table('meetings')
            ->where('id', $id)
            ->update([
                'attachments' => json_encode($attachments),
                'updated_at' => now(),
            ]);

        return response()->json([
            'success' => true,
            'attachments' => $attachments,
        ]);

    } catch (\Exception $e) {
        \Log::error('Upload Error: ' . $e->getMessage());
        return response()->json(['error' => $e->getMessage()], 500);
    }
}

public function deleteAttachment(Request $request, $id)
{
    try {
        $meeting = DB::table('meetings')
            ->where('id', $id)
            ->where('user_id', Auth::id())
            ->first();

        if (!$meeting) {
            return response()->json(['error' => 'Meeting not found'], 404);
        }

        $attachments = json_decode($meeting->attachments, true) ?? [];
        $index = $request->input('index');

        if (isset($attachments[$index])) {
            // Supprimer le fichier physique
            \Storage::disk('public')->delete($attachments[$index]['path']);

            // Retirer du tableau
            array_splice($attachments, $index, 1);

            // Sauvegarder
            DB::table('meetings')
                ->where('id', $id)
                ->update([
                    'attachments' => json_encode($attachments),
                    'updated_at' => now(),
                ]);
        }

        return response()->json([
            'success' => true,
            'attachments' => $attachments,
        ]);

    } catch (\Exception $e) {
        \Log::error('Delete Attachment Error: ' . $e->getMessage());
        return response()->json(['error' => $e->getMessage()], 500);
    }
}

/**
 * 📁 Déterminer le type de fichier
 */
private function getFileType($mimeType)
{
    if (str_contains($mimeType, 'image')) return 'image';
    if (str_contains($mimeType, 'pdf')) return 'pdf';
    if (str_contains($mimeType, 'word') || str_contains($mimeType, 'document')) return 'doc';
    if (str_contains($mimeType, 'sheet') || str_contains($mimeType, 'excel')) return 'excel';
    if (str_contains($mimeType, 'presentation') || str_contains($mimeType, 'powerpoint')) return 'ppt';
    return 'file';
}
}
