<?php

namespace App\Http\Controllers\Student;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Wallet;
use App\Models\WalletTransaction;
use App\Models\PaymentMethod;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class WalletController extends Controller
{
    /**
     * 🔹 Affiche la page du portefeuille étudiant (solde + historique + moyens de paiement)
     */
    public function index()
    {
        $user = Auth::user();

        // 🔸 Récupération ou création du portefeuille de l’utilisateur
        $wallet = Wallet::firstOrCreate(
            ['user_id' => $user->id],
            ['balance' => 0.00, 'currency' => 'TND']
        );

        // 🔸 Liste des méthodes de paiement actives
        $paymentMethods = PaymentMethod::where('is_active', true)->get();

        // 🔸 Historique des transactions
        $transactions = WalletTransaction::with('paymentMethod')
            ->where('user_id', $user->id)
            ->latest()
            ->take(10)
            ->get();

        // 🔸 Points fictifs (ex : pour un système de fidélité)
        $points = 1599;

        return view('student.wallet', [
            'points' => $points,
            'wallet' => $wallet,
            'paymentMethods' => $paymentMethods,
            'transactions' => $transactions,
            'section' => 'Wallet',
            'page' => 'Online Payment',
        ]);
    }

    /**
     * 📤 Traite le formulaire d’upload de preuve de paiement (offline)
     */
    public function uploadProof(Request $request)
    {
        $request->validate([
            'provider' => 'required|string|max:100',
            'amount' => 'required|numeric|min:1',
            'reference' => 'required|string|max:100',
            'receipt' => 'required|file|mimes:jpg,jpeg,png,pdf|max:5120',
        ]);

        $user = Auth::user();
        $method = PaymentMethod::where('name', $request->provider)->first();

        // 🔹 Sauvegarde du fichier reçu
        $filePath = $request->file('receipt')->store('receipts', 'public');

        // 🔹 Enregistrement de la transaction
        $transaction = WalletTransaction::create([
            'user_id' => $user->id,
            'payment_method_id' => $method?->id,
            'amount' => $request->amount,
            'reference' => $request->reference,
            'receipt_path' => $filePath,
            'status' => 'pending',
            'type' => 'credit',
        ]);

        return redirect()
            ->back()
            ->with('success', 'Votre preuve de paiement a été soumise avec succès. Elle sera vérifiée sous 24-48h.');
    }

    /**
     * ✅ Valide une transaction manuellement (admin)
     * ⚠️ À placer dans un contrôleur Admin dans un vrai cas
     */
    public function validateTransaction($id)
    {
        $transaction = WalletTransaction::findOrFail($id);
        $wallet = Wallet::where('user_id', $transaction->user_id)->first();

        if ($transaction->status === 'pending') {
            $transaction->update([
                'status' => 'success',
                'verified_by' => Auth::id(),
            ]);

            // 🔹 Met à jour le solde du portefeuille
            $wallet->balance += $transaction->amount;
            $wallet->save();
        }

        return back()->with('success', 'Transaction validée et solde mis à jour.');
    }

    /**
     * ❌ Refuse une transaction (admin)
     */
    public function rejectTransaction($id)
    {
        $transaction = WalletTransaction::findOrFail($id);

        if ($transaction->status === 'pending') {
            $transaction->update([
                'status' => 'failed',
                'verified_by' => Auth::id(),
            ]);
        }

        return back()->with('info', 'Transaction refusée.');
    }
}
