<?php

namespace App\Http\Controllers\Student;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;

class ProjectController extends Controller
{
    public function index()
    {
        $projects = DB::table('projects')
            ->where('user_id', Auth::id())
            ->orderBy('created_at', 'desc')
            ->get();

        return view('student.projects.index', compact('projects'));
    }

    public function create()
    {
        return view('student.projects.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'type' => 'required|in:react,java,python,html,nodejs,other',
        ]);

        // Créer le dossier du projet
        $folderName = Str::slug($request->name) . '-' . time();
        $folderPath = 'projects/' . Auth::id() . '/' . $folderName;
        $fullPath = storage_path('app/public/' . $folderPath);

        File::makeDirectory($fullPath, 0755, true);

        // Créer un fichier initial selon le type
        $mainFile = $this->createInitialFile($fullPath, $request->type);

        // Enregistrer dans la base de données
        DB::table('projects')->insert([
            'user_id' => Auth::id(),
            'name' => $request->name,
            'description' => $request->description,
            'type' => $request->type,
            'folder_path' => $folderPath,
            'main_file' => $mainFile,
            'status' => 'active',
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        return redirect()->route('student.projects.index')
            ->with('success', ' Project successfully created!');
    }

    public function editor($id)
    {
        $project = DB::table('projects')
            ->where('id', $id)
            ->where('user_id', Auth::id())
            ->first();

        if (!$project) {
            return redirect()->route('student.projects.index')
                ->with('error', 'Projet introuvable');
        }

        // Lire le contenu du fichier principal
        $filePath = storage_path('app/public/' . $project->folder_path . '/' . $project->main_file);
        $content = File::exists($filePath) ? File::get($filePath) : '';

        return view('student.projects.editor', compact('project', 'content'));
    }

    public function updateFile(Request $request, $id)
    {
        $request->validate([
            'content' => 'required|string',
        ]);

        $project = DB::table('projects')
            ->where('id', $id)
            ->where('user_id', Auth::id())
            ->first();

        if (!$project) {
            return response()->json(['error' => 'Projet introuvable'], 404);
        }

        $filePath = storage_path('app/public/' . $project->folder_path . '/' . $project->main_file);
        File::put($filePath, $request->content);

        return response()->json(['success' => true, 'message' => 'Fichier sauvegardé']);
    }

    public function destroy($id)
    {
        $project = DB::table('projects')
            ->where('id', $id)
            ->where('user_id', Auth::id())
            ->first();

        if (!$project) {
            return redirect()->route('student.projects.index')
                ->with('error', 'Projet introuvable');
        }

        // Supprimer le dossier
        $fullPath = storage_path('app/public/' . $project->folder_path);
        File::deleteDirectory($fullPath);

        // Supprimer de la base de données
        DB::table('projects')->where('id', $id)->delete();

        return redirect()->route('student.projects.index')
            ->with('success', 'Projet supprimé');
    }

    private function createInitialFile($folderPath, $type)
    {
        $templates = [
            'react' => ['file' => 'App.jsx', 'content' => "import React from 'react';\n\nfunction App() {\n  return (\n    <div>\n      <h1>Hello React!</h1>\n    </div>\n  );\n}\n\nexport default App;"],
            'java' => ['file' => 'Main.java', 'content' => "public class Main {\n    public static void main(String[] args) {\n        System.out.println(\"Hello Java!\");\n    }\n}"],
            'python' => ['file' => 'main.py', 'content' => "# Mon projet Python\nprint('Hello Python!')"],
            'html' => ['file' => 'index.html', 'content' => "<!DOCTYPE html>\n<html>\n<head>\n    <title>Mon Projet</title>\n</head>\n<body>\n    <h1>Hello World!</h1>\n</body>\n</html>"],
            'nodejs' => ['file' => 'index.js', 'content' => "console.log('Hello Node.js!');"],
            'other' => ['file' => 'main.txt', 'content' => "# Mon projet\n"],
        ];

        $template = $templates[$type] ?? $templates['other'];
        File::put($folderPath . '/' . $template['file'], $template['content']);

        return $template['file'];
    }
}
