<?php

namespace App\Http\Controllers\Student;

use App\Http\Controllers\Controller;
use App\Models\NovaConversation;
use App\Models\NovaMessage;
use Illuminate\Http\Request;
use OpenAI\Laravel\Facades\OpenAI;
use Exception;

class NovaController extends Controller
{
    /**
     * Affiche la page Nova AI.
     */
public function index()
{
    $user = auth()->user();

    // Récupérer les conversations avec le dernier message utilisateur
    $conversations = NovaConversation::where('user_id', $user->id)
        ->with('lastUserMessage') // Eager loading de la relation
        ->orderBy('last_message_at', 'desc')
        ->get();

    return view('student.nova', [
        'title' => 'Nova Chat',
        'section' => 'Nova',
        'page' => 'AI Assistant',
        'points' => 1599,
        'user' => $user,
        'conversations' => $conversations,
    ]);
}

    /**
     * Créer une nouvelle conversation
     */
    public function createConversation(Request $request)
    {
        $conversation = NovaConversation::create([
            'user_id' => auth()->id(),
            'title' => 'New Conversation',
            'last_message_at' => now(),
        ]);

        return response()->json([
            'success' => true,
            'conversation_id' => $conversation->id,
        ]);
    }

    /**
     * Charger une conversation existante
     */
    public function loadConversation($conversationId)
    {
        $conversation = NovaConversation::with('orderedMessages')
            ->where('id', $conversationId)
            ->where('user_id', auth()->id())
            ->firstOrFail();

        return response()->json([
            'success' => true,
            'conversation' => $conversation,
            'messages' => $conversation->orderedMessages,
        ]);
    }

    /**
     * Envoyer un message
     */
  public function message(Request $request)
{
    $request->validate([
        'message' => 'required|string|max:2000',
        'conversation_id' => 'nullable|exists:nova_conversations,id',
        'image' => 'nullable|string',
        'image_name' => 'nullable|string|max:255',
    ]);

    $userMessage = $request->input('message');
    $conversationId = $request->input('conversation_id');
    $imageData = $request->input('image');
    $imageName = $request->input('image_name');

    if (!$conversationId) {
        $conversation = NovaConversation::create([
            'user_id' => auth()->id(),
            'last_message_at' => now(),
        ]);
        $conversationId = $conversation->id;
    } else {
        $conversation = NovaConversation::where('id', $conversationId)
            ->where('user_id', auth()->id())
            ->firstOrFail();
    }

    $history = NovaMessage::where('conversation_id', $conversationId)
        ->orderBy('created_at', 'asc')
        ->take(20)
        ->get();

    $messages = [
        [
            'role' => 'system',
            'content' => 'You are Nova, a friendly AI assistant for an educational platform focused on development and programming. You help students learn to code, debug issues, explain concepts, and provide code examples. You can also analyze images related to programming, code screenshots, diagrams, and UI designs. When sharing code, always specify the programming language. Be encouraging and pedagogical in your responses.'
        ]
    ];

    foreach ($history as $msg) {
        $messages[] = [
            'role' => $msg->role,
            'content' => $msg->content
        ];
    }

    // Construire le message actuel avec ou sans image
    $currentMessage = ['role' => 'user'];

    if ($imageData) {
        // 🔥 FORMAT CORRIGÉ pour GPT-4 Vision
        $currentMessage['content'] = [
            [
                'type' => 'text',
                'text' => $userMessage
            ],
            [
                'type' => 'image_url',
                'image_url' => [
                    'url' => $imageData,
                    'detail' => 'high' // Optionnel : 'low', 'high', ou 'auto'
                ]
            ]
        ];
    } else {
        $currentMessage['content'] = $userMessage;
    }

    $messages[] = $currentMessage;

    try {
        // 🔥 UTILISER LE BON MODÈLE
        // Choisir selon ce qui est disponible dans votre compte OpenAI
        $model = $imageData ? 'gpt-4o' : 'gpt-4-turbo'; // ou 'gpt-4o-mini' pour moins cher

        $apiParams = [
            'model' => $model,
            'messages' => $messages,
            'temperature' => 0.7,
        ];

        // 🔥 max_tokens uniquement si pas d'image (gpt-4o le gère automatiquement)
        if (!$imageData) {
            $apiParams['max_tokens'] = 1500;
        } else {
            $apiParams['max_tokens'] = 2000; // Plus pour l'analyse d'image
        }

        $response = OpenAI::chat()->create($apiParams);

        $novaReply = trim($response['choices'][0]['message']['content'] ?? "Sorry, I couldn't understand that.");

        // Sauvegarder le message
        $userMessageContent = $userMessage;
        if ($imageData && $imageName) {
            $userMessageContent .= " [Image: {$imageName}]";
        }

        NovaMessage::create([
            'conversation_id' => $conversationId,
            'role' => 'user',
            'content' => $userMessageContent,
        ]);

        NovaMessage::create([
            'conversation_id' => $conversationId,
            'role' => 'assistant',
            'content' => $novaReply,
        ]);

        $conversation->updateLastMessageTime();
        $conversation->generateTitle();

        return response()->json([
            'reply' => $novaReply,
            'success' => true,
            'conversation_id' => $conversationId,
            'conversation_title' => $conversation->title,
        ]);

    } catch (Exception $e) {
        // 🔥 LOG L'ERREUR COMPLÈTE pour debugging
        \Log::error('Nova API Error: ' . $e->getMessage());

        return response()->json([
            'reply' => "I'm having trouble connecting right now. Please try again in a moment.",
            'success' => false,
            'error' => $e->getMessage() // Utile pour le debug
        ], 500);
    }
}

    /**
     * Supprimer une conversation
     */
    public function deleteConversation($conversationId)
    {
        $conversation = NovaConversation::where('id', $conversationId)
            ->where('user_id', auth()->id())
            ->firstOrFail();

        $conversation->delete();

        return response()->json([
            'success' => true,
            'message' => 'Conversation deleted successfully'
        ]);
    }

    /**
     * Obtenir la liste des conversations
     */
    public function getConversations()
    {
        $conversations = NovaConversation::where('user_id', auth()->id())
            ->orderBy('last_message_at', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'conversations' => $conversations,
        ]);
    }
}
