<?php

namespace App\Http\Controllers\Student;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Meeting;
use App\Models\Group;
use App\Models\StudentTask;
use App\Models\LiveSession;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class LiveSessionController extends Controller
{
    /**
     * Affiche la page principale des Live Sessions (vue calendrier)
     */
    public function index()
{
    $user = Auth::user();

    $groups = $user->groups()->pluck('groups.id')->toArray();

    $startOfWeek = Carbon::now()->startOfWeek();
    $endOfWeek   = Carbon::now()->endOfWeek()->endOfDay();

    // 1. Meetings
    $meetings = Meeting::with(['group', 'user'])
        ->whereIn('group_id', $groups)
        ->whereBetween('start_time', [$startOfWeek, $endOfWeek])
        ->orderBy('start_time')
        ->get()
        ->map(function ($m) {

            $start = Carbon::parse($m->start_time);
            $end   = Carbon::parse($m->end_time);

            // ajouter champs custom à l'objet Eloquent
            $m->calendar_id     = "meeting_" . $m->id;
            $m->calendar_title  = $m->topic;
            $m->calendar_type   = "class";
            $m->calendar_status = "open";
            $m->calendar_day    = strtolower($start->format('l'));
            $m->calendar_start  = $start->format("H:i");
            $m->calendar_end    = $end->format("H:i");
            $m->calendar_link   = $m->join_url;

            return $m; // ⚠️ garder le modèle Eloquent !
        });

    // 2. Live Sessions
    $otherSessions = LiveSession::with(['group', 'user'])
        ->whereIn('group_id', $groups)
        ->where('week_start_date', $startOfWeek->format("Y-m-d"))
        ->orderBy("day_of_week")
        ->orderBy("start_time")
        ->get()
        ->map(function ($s) {

            $s->calendar_id     = "session_" . $s->id;
            $s->calendar_title  = $s->title;
            $s->calendar_type   = $s->type;
            $s->calendar_status = $s->status;
            $s->calendar_day    = $s->day_of_week;
            $s->calendar_start  = $s->start_time;
            $s->calendar_end    = $s->end_time;
            $s->calendar_link   = $s->link;

            return $s; // ⚠️ garder le modèle Eloquent !
        });

    // 3. Fusionner proprement
    $sessions = $meetings->merge($otherSessions);

    // 4. Tasks
    $tasks = StudentTask::where('user_id', $user->id)
        ->orderBy('completed')
        ->orderByDesc('created_at')
        ->get();

    return view('student.live-sessions', compact('sessions', 'tasks', 'user'));
}


public function update(Request $request, $id)
{
    $session = LiveSession::findOrFail($id);

    // Vérifier que l'utilisateur peut modifier la session
    if ($session->user_id !== Auth::id()) {
        return response()->json(['error' => 'Unauthorized'], 403);
    }

    // Mise à jour live en autosave
    $session->update([
        'title'        => $request->title,
        'start_time'   => $request->start,
        'end_time'     => $request->end,
        'notes'        => $request->notes,
        'subtasks'     => $request->subtasks, // JSON
    ]);

    return response()->json(['success' => true]);
}





    /**
     * Créer une nouvelle tâche
     */
public function storeTask(Request $request)
{
    $task = StudentTask::create([
        'user_id' => Auth::id(),
        'title' => $request->title ?? '',
        'completed' => false,
    ]);

    return response()->json(['id' => $task->id]);
}

public function move(Request $request, $id)
{
    $session = LiveSession::findOrFail($id);

    // Vérifier que l'utilisateur peut modifier cette session
    if ($session->user_id !== Auth::id()) {
        return response()->json(['error' => 'Unauthorized'], 403);
    }

    // Calculer la nouvelle heure de fin basée sur la durée originale
    $originalStart = Carbon::createFromFormat('H:i', substr($session->start_time, 0, 5));
    $originalEnd = Carbon::createFromFormat('H:i', substr($session->end_time, 0, 5));
    $durationMinutes = $originalStart->diffInMinutes($originalEnd);

    $newStart = Carbon::createFromFormat('H:i', $request->start_time);
    $newEnd = $newStart->copy()->addMinutes($durationMinutes);

    $session->update([
        'day_of_week' => $request->day_of_week,
        'start_time' => $newStart->format('H:i'),
        'end_time' => $newEnd->format('H:i'),
    ]);

    return response()->json(['success' => true]);
}

public function updateTask(Request $request, $id)
{
    $task = StudentTask::findOrFail($id);

    if ($task->user_id !== Auth::id()) {
        return response()->json(['error' => 'Unauthorized'], 403);
    }

    $task->title = $request->title ?? '';
    $task->save();

    return response()->json(['success' => true]);
}

    /**
     * Marquer une tâche "To Do" comme complétée ou non
     */
    public function toggleTask(Request $request, $id)
    {
        $task = StudentTask::findOrFail($id);

        if ($task->user_id !== Auth::id()) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $task->completed = !$task->completed;
        $task->save();

        return response()->json(['completed' => $task->completed]);
    }

    /**
     * Créer une nouvelle session (project, break, lunch, meeting)
     */
    public function store(Request $request)
    {
        $request->validate([
            'group_id'   => 'required|exists:groups,id',
            'title'      => 'required|string|max:255',
            'type'       => 'required|in:project,break,meeting,lunch',
            'day_of_week'=> 'required|in:monday,tuesday,wednesday,thursday,friday,saturday,sunday',
            'start_time' => 'required',
            'end_time'   => 'required',
        ]);

        // 🔹 Calculer la date de début de semaine actuelle
        $weekStartDate = Carbon::now()->startOfWeek()->format('Y-m-d');

        LiveSession::create([
            'group_id'      => $request->group_id,
            'user_id'       => Auth::id(),
            'title'         => $request->title,
            'type'          => $request->type,
            'status'        => $request->status ?? 'open',
            'day_of_week'   => $request->day_of_week,
            'week_start_date' => $weekStartDate, // 🎯 Important !
            'start_time'    => $request->start_time,
            'end_time'      => $request->end_time,
            'color'         => $request->color ?? '#A5D6A7',
            'link'          => $request->link,
        ]);

        return redirect()->back()->with('success', 'Session created successfully!');
    }

    public function destroy($id)
{
    $session = LiveSession::findOrFail($id);

    if ($session->user_id !== Auth::id()) {
        return response()->json(['error' => 'Unauthorized'], 403);
    }

    $session->delete();

    return response()->json(['success' => true]);
}


public function destroyTask($id)
{
    $task = StudentTask::findOrFail($id);

    if ($task->user_id !== Auth::id()) {
        return response()->json(['error' => 'Unauthorized'], 403);
    }

    $task->delete();

    return response()->json(['success' => true]);
}




}
