<?php

namespace App\Http\Controllers\Student;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\LiveSession;
use App\Models\LiveMessage;
use App\Models\LiveParticipant;

class LiveController extends Controller
{
    /**
     * Liste des sessions live disponibles pour l'étudiant
     */
    public function index()
    {
        $sessions = LiveSession::with('teacher')
            ->orderBy('day_of_week')
            ->get();

        return view('student.live.index', compact('sessions'));
    }

    public function getMessages($id)
{
    $messages = \App\Models\LiveMessage::with('sender')
        ->where('session_id', $id)
        ->latest()
        ->take(50)
        ->get()
        ->reverse()
        ->values();

    return response()->json($messages);
}

    /**
     * Affiche une session live spécifique (vue principale)
     */
    public function show($id)
    {
        $session = LiveSession::with(['teacher', 'messages.sender', 'participants.user'])
            ->findOrFail($id);

        // Si l'utilisateur rejoint la session, on l'ajoute comme participant s'il n'y est pas déjà
        if (Auth::check()) {
            LiveParticipant::firstOrCreate(
                [
                    'session_id' => $session->id,
                    'user_id' => Auth::id(),
                ],
                [
                    'role' => Auth::id() === $session->teacher_id ? 'teacher' : 'student',
                    'joined_at' => now(),
                ]
            );
        }

        return view('student.live.show', compact('session'));
    }

    /**
     * Envoi d’un message dans le chat en temps réel
     */
public function sendMessage(Request $request, $id)
{
    $request->validate([
        'message' => 'required|string|max:1000',
    ]);

    $session = LiveSession::findOrFail($id);

    $message = LiveMessage::create([
        'session_id' => $session->id,
        'sender_id' => Auth::id(),
        'message' => $request->message,
        'sent_at' => now(),
    ]);

    return response()->json([
        'success' => true,
        'message' => [
            'id' => $message->id,
            'sender' => Auth::user()->name ?? 'Utilisateur',
            'message' => $message->message,
        ],
    ]);
}


    /**
     * Marque le départ d’un participant lorsqu’il quitte la session
     */
    public function leave($id)
    {
        if (!Auth::check()) {
            return response()->json(['error' => 'Unauthorized'], 401);
        }

        $participant = LiveParticipant::where('session_id', $id)
            ->where('user_id', Auth::id())
            ->first();

        if ($participant) {
            $participant->update(['left_at' => now()]);
        }

        return response()->json(['success' => true]);
    }
}
