<?php

namespace App\Http\Controllers\Student;

use App\Http\Controllers\Controller;
use Google\Client;
use Google\Service\Calendar;
use Google\Service\Calendar\Event;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class GoogleMeetController extends Controller
{
    // ✅ Liste des réunions existantes
    public function index()
    {
        $meetings = DB::table('meetings')
            ->where('user_id', Auth::id())
            ->orderBy('start_time', 'desc')
            ->get();

        return view('student.meet.index', compact('meetings'));
    }

    // ✅ Création automatique d'une réunion Google Meet
  public function create(Request $request)
{
    try {
        // 1️⃣ Config Google
        $client = new \Google\Client();
        $client->setAuthConfig(storage_path('app/google-service-account.json'));
        $client->addScope(\Google\Service\Calendar::CALENDAR);
        $service = new \Google\Service\Calendar($client);

        // 2️⃣ Détails de la réunion
        $start = date('c', strtotime('+5 minutes'));
        $end   = date('c', strtotime('+65 minutes'));

        $event = new \Google\Service\Calendar\Event([
            'summary' => 'Classe en direct - ' . Auth::user()->name,
            'description' => 'Cours en ligne via Google Meet',
            'start' => ['dateTime' => $start, 'timeZone' => 'Africa/Tunis'],
            'end'   => ['dateTime' => $end, 'timeZone' => 'Africa/Tunis'],
            'conferenceData' => [
                'createRequest' => [
                    'requestId' => uniqid(),
                ],
            ],
        ]);

        // 3️⃣ Création initiale
        $event = $service->events->insert('primary', $event, ['conferenceDataVersion' => 1]);

        // 4️⃣ Boucle d’attente jusqu’à ce que le lien Meet soit disponible
        $meetLink = null;
        $retries = 0;

        while ($retries < 10 && !$meetLink) {
            sleep(1); // attendre 1 seconde
            $fetched = $service->events->get('primary', $event->id);
            $meetLink = $fetched->hangoutLink ?? null;
            $retries++;
        }

        if (!$meetLink) {
            throw new \Exception("Google Meet n’a pas encore généré de lien après 10 secondes.");
        }

        // 5️⃣ Enregistrer en base
        DB::table('meetings')->insert([
            'user_id' => Auth::id(),
            'title' => $event->getSummary(),
            'meet_link' => $meetLink,
            'start_time' => $start,
            'end_time' => $end,
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        return redirect()->route('student.meet.index')
            ->with('success', '✅ Réunion Google Meet créée avec succès !');
    }
    catch (\Google\Service\Exception $e) {
        \Log::error('❌ Erreur Google Meet API', ['message' => $e->getMessage()]);
        return back()->with('error', 'Erreur Google API : ' . $e->getMessage());
    }
    catch (\Exception $e) {
        \Log::error('❌ Erreur inattendue Google Meet', ['message' => $e->getMessage()]);
        return back()->with('error', $e->getMessage());
    }
}
}
