<?php

namespace App\Http\Controllers\Student;

use App\Http\Controllers\Controller;
use App\Models\Goal;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class GoalsController extends Controller
{
    /**
     * Afficher la page Goals avec toutes les tâches
     */
    public function index()
    {
        $user = Auth::user();

        // Récupérer les goals par statut
        $notStarted = Goal::where('user_id', $user->id)
            ->byStatus('not_started')
            ->get();

        $inProgress = Goal::where('user_id', $user->id)
            ->byStatus('in_progress')
            ->get();

        $done = Goal::where('user_id', $user->id)
            ->byStatus('done')
            ->get();

        // Calculer les progrès
        $totalGoals = Goal::where('user_id', $user->id)->count();
        $completedGoals = $done->count();

        $todayProgress = $totalGoals > 0 ? round(($completedGoals / $totalGoals) * 100) : 0;
        $weekProgress = $totalGoals > 0 ? round((($completedGoals + $inProgress->count()) / $totalGoals) * 100) : 0;
        $monthProgress = $totalGoals > 0 ? round(($completedGoals / $totalGoals) * 80) : 0;

        return view('student.goals', compact(
            'notStarted',
            'inProgress',
            'done',
            'todayProgress',
            'weekProgress',
            'monthProgress'
        ));
    }

    /**
     * Créer une nouvelle tâche
     */
public function store(Request $request)
{
    $request->validate([
        'title' => 'nullable|string|max:255',
        'status' => 'required|in:not_started,in_progress,done',
        'due_date' => 'nullable|string',
        'label' => 'nullable|string',
    ]);

    $goal = Goal::create([
        'user_id'    => Auth::id(),
        'title'      => $request->title,
        'description'=> $request->description,
        'notes'      => $request->notes,
        'checklist'  => $request->checklist,
        'comment'    => $request->comment,
        'status'     => $request->status,
        'priority'   => $request->priority,
        'task_type'  => $request->task_type,
        'due_date'   => $request->due_date,
        'label'      => $request->label,
        'position'   => Goal::where('user_id', Auth::id())->where('status', $request->status)->count(),
    ]);

    return response()->json([
        'success' => true,
        'goal'    => $goal
    ]);
}

    /**
     * Mettre à jour une tâche
     */
 public function update(Request $request, $id)
{
    $goal = Goal::where('user_id', Auth::id())->findOrFail($id);

 $goal->update([
    'title'       => $request->title,
    'description' => $request->description,
    'comment'     => $request->comment,
    'notes'       => $request->notes,
    'checklist'   => $request->checklist,
    'priority'    => $request->priority,
    'task_type'   => $request->task_type,
    'due_date'    => $request->due_date,
    'status'      => $request->status,     // ✅ AJOUT OBLIGATOIRE
    'label'       => $request->label,
]);

    return response()->json([
        'success' => true,
        'goal' => $goal
    ]);
}

    /**
     * Supprimer une tâche
     */
    public function destroy($id)
    {
        $goal = Goal::where('user_id', Auth::id())->findOrFail($id);
        $goal->delete();

        return response()->json([
            'success' => true
        ]);
    }

    /**
     * Déplacer une tâche vers une autre colonne
     */
    public function move(Request $request, $id)
    {
        $request->validate([
            'status' => 'required|in:not_started,in_progress,done',
        ]);

        $goal = Goal::where('user_id', Auth::id())->findOrFail($id);

        $goal->update([
            'status' => $request->status,
            'position' => Goal::where('user_id', Auth::id())->where('status', $request->status)->count()
        ]);

        return response()->json([
            'success' => true,
            'goal' => $goal
        ]);
    }

  public function table()
{
    $tasks = Goal::with('subtasks')->get();
    return view('student.goals-table', compact('tasks'));
}

public function reorder(Request $request)
{
    $request->validate([
        'status' => 'required|in:not_started,in_progress,done',
        'order' => 'required|array'
    ]);

    foreach ($request->order as $index => $goalId) {
        Goal::where('user_id', Auth::id())
            ->where('id', $goalId)
            ->update(['position' => $index]);
    }

    return response()->json(['success' => true]);
}
}
