<?php

namespace App\Http\Controllers\Student;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;

class AccountController extends Controller
{
    /**
     * Affiche la page du compte avec toutes les données
     */
    public function index()
    {
        $user = Auth::user();
        return view('student.account', compact('user'));
    }

    /**
     * Met à jour les informations personnelles
     */
    public function updatePersonalInfo(Request $request)
    {
        $request->validate([
            'name'           => 'required|string|max:255',
            'first_name'     => 'required|string|max:255',
            'last_name'      => 'required|string|max:255',
            'date_of_birth'  => 'required|date',
            'gender'         => 'required|in:Male,Female,Other',
            'phone'          => 'nullable|string|max:20',
            'second_phone'   => 'nullable|string|max:20',
            'governorate'    => 'nullable|string|max:100',
            'city'           => 'nullable|string|max:100',
            'postal_code'    => 'nullable|string|max:10',
            'street_address' => 'nullable|string|max:255',
        ]);

        $user = Auth::user();

        $user->update([
            'name'           => $request->name,
            'first_name'     => $request->first_name,
            'last_name'      => $request->last_name,
            'date_of_birth'  => $request->date_of_birth,
            'gender'         => $request->gender,
            'phone'          => $request->phone,
            'second_phone'   => $request->second_phone,
            'governorate'    => $request->governorate,
            'city'           => $request->city,
            'postal_code'    => $request->postal_code,
            'street_address' => $request->street_address,
        ]);

        return redirect()->route('student.account')->with('success', 'Personal information updated successfully ✅');
    }

    /**
     * Envoie le code de vérification par email
     */
    public function sendEmailVerification()
    {
        $user = Auth::user();
        $code = rand(100000, 999999);

        $user->update(['verification_code' => $code]);

        // Envoyer l'email avec le code
        Mail::raw("Your verification code is: {$code}", function ($message) use ($user) {
            $message->to($user->email)
                    ->subject('Email Verification Code');
        });

        return redirect()->route('student.account')->with([
            'success' => 'Verification code sent to your email',
            'open_code_modal' => true
        ]);
    }

    /**
     * Confirme et change l'email
     */
    public function confirmEmailChange(Request $request)
    {
        $request->validate([
            'code'      => 'required|numeric',
            'new_email' => 'required|email|unique:users,email',
        ]);

        $user = Auth::user();

        if ($user->verification_code != $request->code) {
            return redirect()->route('student.account')->with([
                'error' => 'Invalid verification code',
                'open_code_modal' => true
            ]);
        }

        $user->update([
            'email' => $request->new_email,
            'verification_code' => null,
        ]);

        return redirect()->route('student.account')->with('success', 'Email updated successfully ✅');
    }

    /**
     * Change le mot de passe
     */
    public function updatePassword(Request $request)
    {
        $request->validate([
            'current_password'          => 'required',
            'new_password'              => 'required|min:8|confirmed',
            'new_password_confirmation' => 'required',
        ]);

        $user = Auth::user();

        if (!Hash::check($request->current_password, $user->password)) {
            return redirect()->route('student.account')->with('error', 'Current password is incorrect');
        }

        $user->update([
            'password' => Hash::make($request->new_password),
        ]);

        return redirect()->route('student.account')->with('success', 'Password updated successfully ✅');
    }
}
