<?php

namespace App\Http\Controllers\Affiliate;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Wallet;
use App\Models\WalletTransaction;
use App\Models\PaymentMethod;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class AffiliateTransactionController extends Controller
{
    /**
     * 🔹 Affiche la page portefeuille affilié + stats + transactions + méthodes de paiement
     */
    public function index()
    {
        $user = Auth::user();

        // 🔸 Wallet affilié (créé automatiquement si non existant)
        $wallet = Wallet::firstOrCreate(
            ['user_id' => $user->id],
            ['balance' => 0.00, 'currency' => 'TND']
        );

        // 🔸 Méthodes de paiement disponibles
        $paymentMethods = PaymentMethod::where('is_active', true)->get();

        // 🔸 Historique des transactions affiliées
        $transactions = WalletTransaction::with('paymentMethod')
            ->where('user_id', $user->id)
            ->orderBy('created_at', 'desc')
            ->take(25)
            ->get();

        // 🔸 Statistiques (calculées automatiquement)
        $stats = [
            'total_revenue' => WalletTransaction::where('user_id', $user->id)->where('type', 'credit')->sum('amount') . ' DT',
            'month_revenue' => WalletTransaction::where('user_id', $user->id)
                                ->whereMonth('created_at', now()->month)
                                ->where('type', 'credit')
                                ->sum('amount') . ' DT',
            'teaching_hours' => 0, // pas applicable → tu peux supprimer si inutile
            'balance' => $wallet->balance . ' DT',
        ];

        return view('affiliate.transaction', [
            'wallet'          => $wallet,
            'paymentMethods'  => $paymentMethods,
            'transactions'    => $transactions,
            'stats'           => $stats,
            'section'         => 'Wallet',
            'page'            => 'Transactions',
        ]);
    }

    /**
     * 📤 Upload d'une preuve de paiement (si l'affilié souhaite créditer son wallet)
     */
    public function uploadProof(Request $request)
    {
        $request->validate([
            'provider'  => 'required|string|max:100',
            'amount'    => 'required|numeric|min:1',
            'reference' => 'required|string|max:100',
            'receipt'   => 'required|file|mimes:jpg,jpeg,png,pdf|max:5120',
        ]);

        $user = Auth::user();
        $method = PaymentMethod::where('name', $request->provider)->first();

        // 🔹 Upload de fichier
        $filePath = $request->file('receipt')->store('receipts', 'public');

        // 🔹 Transaction en attente de validation
        WalletTransaction::create([
            'user_id'            => $user->id,
            'payment_method_id'  => $method?->id,
            'amount'             => $request->amount,
            'reference'          => $request->reference,
            'receipt_path'       => $filePath,
            'status'             => 'pending',
            'type'               => 'credit',
        ]);

        return back()->with('success', 'Votre preuve de paiement a été envoyée. Vérification sous 24-48h.');
    }

    /**
     * 💸 Soumettre une demande de retrait
     */
    public function withdraw(Request $request)
    {
        $request->validate([
            'method' => 'required|string|min:3',
            'amount' => 'required|numeric|min:10',
        ]);

        $user = Auth::user();
        $wallet = Wallet::where('user_id', $user->id)->first();

        if (!$wallet || $wallet->balance < $request->amount) {
            return back()->with('error', 'Solde insuffisant pour effectuer ce retrait.');
        }

        // 🔹 Enregistrement du retrait
        WalletTransaction::create([
            'user_id' => $user->id,
            'amount'  => $request->amount,
            'status'  => 'pending',
            'type'    => 'withdraw',
            'reference' => 'WDR-' . strtoupper(uniqid()),
            'details' => 'Withdraw via ' . $request->method,
        ]);

        // 🔹 Déduire du solde (attente vérification)
        $wallet->balance -= $request->amount;
        $wallet->save();

        return back()->with('success', 'Votre demande de retrait a été envoyée. Elle sera traitée rapidement.');
    }

    /**
     * 🔎 (Admin) Valider un retrait
     */
    public function validateWithdraw($id)
    {
        $transaction = WalletTransaction::findOrFail($id);

        if ($transaction->type === 'withdraw' && $transaction->status === 'pending') {
            $transaction->update(['status' => 'success']);
        }

        return back()->with('success', 'Retrait validé.');
    }

    /**
     * 🔎 (Admin) Refuser un retrait
     */
    public function rejectWithdraw($id)
    {
        $transaction = WalletTransaction::findOrFail($id);

        if ($transaction->type === 'withdraw' && $transaction->status === 'pending') {

            // 🔹 Remboursement du wallet
            $wallet = Wallet::where('user_id', $transaction->user_id)->first();
            $wallet->balance += $transaction->amount;
            $wallet->save();

            // 🔹 Marquer comme refusé
            $transaction->update(['status' => 'failed']);
        }

        return back()->with('error', 'Retrait refusé.');
    }
}
